/*
Assignment #3 - You need to create an express HTTP server in Node.js which will handle the logic of a todo list app.

- Don't use any database, just store all the data in an array to store the todo list data (in-memory)
    Each todo has a title and a description. The title is a string and the description is a string.
    Each todo should also get an unique autogenerated id every time it is created
   
   The expected API endpoints are defined below,

1. GET `/todos` - Retrieve all todo items
   Description: Returns a list of all todo items.
   Response: 200 OK with an array of todo items in JSON format.
   Example: GET http://localhost:3000/todos

2. GET `/todos/:id` - Retrieve a specific todo item by ID
   Description: Returns a specific todo item identified by its ID.
   Response: 200 OK with the todo item in JSON format if found, or 404 Not Found if not found.
   Example: GET http://localhost:3000/todos/123

3. POST `/todos` - Create a new todo item
   Description: Creates a new todo item.
   Request Body: JSON object representing the todo item.
   Response: 201 Created with the ID of the created todo item in JSON format. eg: {id: 1}
   Example: POST http://localhost:3000/todos
   Request Body: { "title": "Buy groceries", "completed": false, description: "I should buy groceries" }

4. PUT `/todos/:id` - Update an existing todo item by ID
   Description: Updates an existing todo item identified by its ID.
   Request Body: JSON object representing the updated todo item.
   Response: 200 OK if the todo item was found and updated, or 404 Not Found if not found.
   Example: PUT http://localhost:3000/todos/123
   Request Body: { "title": "Buy groceries", "completed": true }

5. DELETE `/todos/:id` - Delete a todo item by ID
   Description: Deletes a todo item identified by its ID.
   Response: 200 OK if the todo item was found and deleted, or 404 Not Found if not found.
   Example: DELETE http://localhost:3000/todos/123
   
- For any other route not defined in the server return 404
*/


// import express module using require function and store it in express variable
const express = require("express");

// create an express application using express function
const app = express();

// Middleware to parse JSON data in the request body
aap.use(express.json());

// create an array to store the todo items
let todos = [];

// Route to get all todo items
app.get("/todos", (req, res) => {
    res.json(todos);
});

// Route to get a specific todo item by ID
app.get("/todos/:id", (req, res) => {
    // get the id from the request parameters and store it in id
    const id = parseInt(req.params.id);

    // find the todo item with the given id
    const todo = todos.find((todo) => todo.id === id);

    // If the todo item is found, send it as a response
    if (todo) {
        res.json(todo);
    } else {
        // If the todo item is not found, send a 404 status code
        res.status(404).send("Todo item not found with id " + id);
    }
});

// Route to create a new todo item
app.post("/todos", (req, res) => {
    // get the title and description from the request body
    // const title = req.body.title;
    // const description = req.body.description;
    const { title, description } = req.body;

    // create a new todo item with an unique id
    const todo = {
        id: todos.length + 1,
        title: title,
        description: description
    };

    // add the new todo item to the todos array
    todos.push(todo);

    // send the todo item as a response with status code 201
    res.status(201).json(todo);
});

// Route to update an existing todo item by ID
app.put("/todos/:id", (req, res) => {
    // get the id from the request parameters and store it in id
    const id = parseInt(req.params.id);

    // find the todo item with the given id
    const todo = todos.find((todo) => todo.id === id);

    // If the todo item is found, update it with the new data
    if (todo) {
        // get the title and description from the request body
        const { title, description } = req.body;

        // update the todo item with the new data
        todo.title = title;
        todo.description = description;

        // send a response with the updated todo item
        res.json(todo);
    } else {
        // If the todo item is not found, send a 404 status code
        res.status(404).send("Todo item not found with id " + id);
    }
});

// Route to delete a todo item by ID
app.delete("/todos/:id", (req, res) => {
    // get the id from the request parameters and store it in id
    const id = parseInt(req.params.id);

    // find the index of the todo item with the given id
    const index = todos.findIndex((todo) => todo.id === id);

    // If the todo item is found, delete it from the array
    if (index !== -1) {
        // remove the todo item from the array
        todos.splice(index, 1);

        // send a response with status code 200
        res.status(200).send("Todo deleted successfully with id " + id);
    } else {
        // If the todo item is not found, send a 404 status code
        res.status(404).send("Todo item not found with id " + id);
    }
});

// Route to handle any other route not defined in the server
/*app.use((req, res) => {
    res.status(404).send("Route Not Found");
});*/
app.all("*", (req, res) => {
    res.status(404).send("Route Not Found");
});

// Start the server on port 3000
app.listen(3000);
